import AsyncStorage from '@react-native-async-storage/async-storage';
import moment from 'moment';
import {Linking} from 'react-native';
import Toast from 'react-native-toast-message';
const storeData = async (storageKey, value) => {
  try {
    const jsonValue = JSON.stringify(value);
    await AsyncStorage.setItem(storageKey, jsonValue);
  } catch (e) {
    // saving error
  }
};

const getData = async storageKey => {
  try {
    const jsonValue = await AsyncStorage.getItem(storageKey);
    return jsonValue != null ? JSON.parse(jsonValue) : {};
  } catch (e) {
    // error reading value
  }
};
const removeData = async storageKey => {
  try {
    await AsyncStorage.removeItem(storageKey);
  } catch (e) {
    // error reading value
  }
};

// const startTime = new Date('2023-03-16T12:55:00+07:00');
const calculateElapsedSeconds = (eventDate, eventTime) => {
  const eventDateTime = moment(
    `${eventDate.split('T')[0]}T${eventTime ?? '13:00:00'}`,
  );
  const now = moment();
  const elapsedSeconds = Math.max(eventDateTime.diff(now, 'seconds'), 0) - 300;
  return elapsedSeconds <= 0 ? 0 : elapsedSeconds;
};

/**
 * Nested Object or Array data value retriever
 * @param source
 * @param variables {string}
 * @param fallbackValue {any}
 * @param allowNull
 * @returns {any|boolean|string}
 */
const getValues = (
  source,
  variables = '',
  fallbackValue,
  allowNull = false,
) => {
  const targetValueHierarchy = (variables || '')
    .toString()
    .replace(/[[\]]/g, '.')
    .split('.')
    .filter(key => key !== '');

  if (source === null && allowNull && targetValueHierarchy.length === 0) {
    return null;
  }

  // Check for string type because string is subtype of Array
  // Don't worry, if the data type not an object or array will fail after that.
  if (!source || ['string, boolean'].includes(typeof source)) {
    return fallbackValue;
  }

  // Retain data type cause data type is dynamic
  let result = Object.assign(source);

  for (let i = 0; i < targetValueHierarchy.length; i++) {
    result = result[targetValueHierarchy[i]];

    if (result === undefined) {
      break;
    }

    if (result === null && i !== targetValueHierarchy.length - 1) {
      result = undefined;
      break;
    }
  }

  if (result === null) {
    return allowNull ? result : fallbackValue;
  }

  return result !== undefined ? result : fallbackValue;
};

const formatLinkYoutube = url => {
  var regExp =
    /^.*((youtu.be\/)|(v\/)|(\/u\/\w\/)|(embed\/)|(watch\?))\??v?=?([^#&?]*).*/;
  var match = url.match(regExp);
  return match && match[7].length == 11 ? match[7] : '';
};
const handleOpenUrl = async url => {
  // Checking if the link is supported for links with custom URL scheme.
  const supported = await Linking.canOpenURL(url);
  if (supported) {
    // Opening the link with some app, if the URL scheme is "http" the web link should be opened
    // by some browser in the mobile
    await Linking.openURL(url);
  }
};

const formatCurrency = price => {
  if (!price || !Number(price)) return;
  return price.toLocaleString('vi', {style: 'currency', currency: 'VND'});
};

const callToast = (typeText, title, message) => {
  return Toast.show({
    type: typeText,
    text1: title,
    text2: message,
    visibilityTime: 2500,
  });
};

const isAdmin = userInfo => {
  if (!userInfo) {
    if (userInfo?.user_permisions[0]?.permision?.code === 'SITE_ADMIN') {
      return true;
    }
  }

  return false;
};

const uploadFile = file => {
  const formData = new FormData();
  formData.append('file', file);
  return formData;
};

const formatDateDDMMYYYY = date => {
  return moment(date).format('DD/MM/YYYY');
};
const formatDateHHMM = date => {
  return moment(date).format('HH:MM');
};

function getToday() {
  return new Date().toISOString().split('T')[0];
}

function splitStringLength(string, length) {
  if (!string) {
    return string;
  }
  const stringLength = string.length;
  if (stringLength > length) {
    return string.slice(0, length) + '...';
  } else {
    return string;
  }
}

function convertToJson(data) {
  try {
    return JSON.parse(data);
  } catch (error) {
    console.error('Error parsing JSON:', error);
    return '';
  }
}

const Utils = {
  convertToJson,
  splitStringLength,
  storeData,
  getData,
  removeData,
  calculateElapsedSeconds,
  getValues,
  formatLinkYoutube,
  handleOpenUrl,
  formatCurrency,
  callToast,
  uploadFile,
  formatDateDDMMYYYY,
  isAdmin,
  getToday,
  formatDateHHMM,
};
export default Utils;
